<?php
/**
 * Plugin Name: Lang Translate API
 * Plugin URI: https://lang.amberdev.online
 * Description: Conecta tu sitio WordPress con la API de traducción Lang (lang.amberdev.online). Traduce texto y HTML mediante shortcodes o funciones PHP.
 * Version: 1.0.0
 * Author: AmberDev
 * Author URI: https://amberdev.online
 * License: GPL v2 or later
 * Text Domain: lang-translate-api
 */

if (!defined('ABSPATH')) {
    exit;
}

define('LANG_TRANSLATE_API_VERSION', '1.0.0');

/**
 * Obtiene la URL base de la API (sin barra final).
 */
function lang_translate_api_base_url() {
    return rtrim(get_option('lang_translate_api_url', 'https://lang.amberdev.online'), '/');
}

/**
 * Obtiene la API key configurada.
 */
function lang_translate_api_key() {
    return get_option('lang_translate_api_key', '');
}

/**
 * Traduce texto usando la API.
 *
 * @param string $text   Texto a traducir.
 * @param string $target Código idioma destino (es, en, etc.).
 * @param string $source Código idioma origen (auto, en, es, etc.). Por defecto 'auto'.
 * @return string Texto traducido o el original en caso de error.
 */
function lang_translate_api_translate($text, $target = 'es', $source = 'auto') {
    $text = trim($text);
    if ($text === '') {
        return '';
    }

    $base = lang_translate_api_base_url();
    $key  = lang_translate_api_key();
    if ($key === '') {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return $text . ' [Lang API: no API key]';
        }
        return $text;
    }

    $url = $base . '/translate';
    $body = wp_json_encode([
        'text'        => $text,
        'target_lang' => $target,
        'source_lang' => $source,
    ]);

    $response = wp_remote_post($url, [
        'timeout' => 30,
        'headers' => [
            'Content-Type' => 'application/json',
            'X-Api-Key'   => $key,
        ],
        'body'    => $body,
    ]);

    if (is_wp_error($response)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return $text . ' [Lang API error: ' . $response->get_error_message() . ']';
        }
        return $text;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $body_response = wp_remote_retrieve_body($response);
            return $text . ' [Lang API ' . $code . ': ' . substr($body_response, 0, 80) . ']';
        }
        return $text;
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (isset($data['translated_text'])) {
        return $data['translated_text'];
    }

    return $text;
}

/**
 * Traduce HTML usando la API.
 *
 * @param string $html   HTML a traducir.
 * @param string $target Código idioma destino (es, en, etc.).
 * @param string $source Código idioma origen (opcional).
 * @return string HTML traducido o el original en caso de error.
 */
function lang_translate_api_translate_html($html, $target = 'es', $source = '') {
    $html = trim($html);
    if ($html === '') {
        return '';
    }

    $base = lang_translate_api_base_url();
    $key  = lang_translate_api_key();
    if ($key === '') {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return $html . '<!-- Lang API: no API key -->';
        }
        return $html;
    }

    $url = $base . '/translate-html';
    $body = wp_json_encode([
        'html'        => $html,
        'target_lang' => $target,
        'source_lang' => $source ?: null,
    ]);

    $response = wp_remote_post($url, [
        'timeout' => 120,
        'headers' => [
            'Content-Type' => 'application/json',
            'X-Api-Key'   => $key,
        ],
        'body'    => $body,
    ]);

    if (is_wp_error($response)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return $html . '<!-- Lang API error: ' . esc_html($response->get_error_message()) . ' -->';
        }
        return $html;
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return $html;
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (isset($data['html'])) {
        return $data['html'];
    }

    return $html;
}

/**
 * Shortcode [lang_translate content="Texto" target="es" source="en"]
 * o [lang_translate target="es"]Texto aquí[/lang_translate]
 */
function lang_translate_api_shortcode($atts, $content = null) {
    $atts = shortcode_atts([
        'content' => '',
        'target'  => 'es',
        'source'  => 'auto',
    ], $atts, 'lang_translate');

    $text = $content !== null ? trim($content) : $atts['content'];
    if ($text === '') {
        return '';
    }

    return lang_translate_api_translate($text, $atts['target'], $atts['source']);
}

add_shortcode('lang_translate', 'lang_translate_api_shortcode');

/**
 * Shortcode [lang_translate_html target="es"]<p>Hello</p>[/lang_translate_html]
 */
function lang_translate_api_shortcode_html($atts, $content = null) {
    $atts = shortcode_atts([
        'target' => 'es',
        'source' => '',
    ], $atts, 'lang_translate_html');

    $html = $content !== null ? trim($content) : '';
    if ($html === '') {
        return '';
    }

    return lang_translate_api_translate_html($html, $atts['target'], $atts['source']);
}

add_shortcode('lang_translate_html', 'lang_translate_api_shortcode_html');

/**
 * Widget flotante: selector de idioma ES | EN que traduce la página automáticamente.
 */
function lang_translate_api_floating_widget_script() {
    if (get_option('lang_translate_api_floating_widget') !== '1') {
        return;
    }
    $key = lang_translate_api_key();
    if ($key === '') {
        return;
    }
    $base = lang_translate_api_base_url();
    $url = $base . '/static/widget.js';
    wp_enqueue_script(
        'lang-translate-widget',
        $url,
        [],
        LANG_TRANSLATE_API_VERSION,
        true
    );
    wp_add_inline_script('lang-translate-widget', '', 'before');
}

add_action('wp_enqueue_scripts', 'lang_translate_api_floating_widget_script');

function lang_translate_api_widget_script_attributes($tag, $handle, $src) {
    if ($handle !== 'lang-translate-widget') {
        return $tag;
    }
    $key = lang_translate_api_key();
    $base = lang_translate_api_base_url();
    $tag = str_replace(' src=', ' data-lang-api-key="' . esc_attr($key) . '" data-lang-api-url="' . esc_attr($base) . '" data-lang-default="es" src=', $tag);
    return $tag;
}

add_filter('script_loader_tag', 'lang_translate_api_widget_script_attributes', 10, 3);

/**
 * Página de ajustes en el menú.
 */
function lang_translate_api_add_menu() {
    add_options_page(
        'Lang Translate API',
        'Lang Translate API',
        'manage_options',
        'lang-translate-api',
        'lang_translate_api_settings_page'
    );
}

add_action('admin_menu', 'lang_translate_api_add_menu');

/**
 * Registrar opciones.
 */
function lang_translate_api_register_settings() {
    register_setting('lang_translate_api', 'lang_translate_api_url', [
        'type'              => 'string',
        'sanitize_callback' => 'esc_url_raw',
    ]);
    register_setting('lang_translate_api', 'lang_translate_api_key', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('lang_translate_api', 'lang_translate_api_floating_widget', [
        'type'              => 'string',
        'sanitize_callback' => function ($v) { return ($v === '1' || $v === true) ? '1' : '0'; },
    ]);
}

add_action('admin_init', 'lang_translate_api_register_settings');

/**
 * Página de configuración.
 */
function lang_translate_api_settings_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    if (isset($_GET['settings-updated'])) {
        add_settings_error(
            'lang_translate_api_messages',
            'lang_translate_api_message',
            __('Configuración guardada.', 'lang-translate-api'),
            'success'
        );
    }

    settings_errors('lang_translate_api_messages');
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <p>Conecta este sitio con la API de traducción <strong>Lang</strong>. Crea una credencial API en <a href="https://lang.amberdev.online" target="_blank" rel="noopener">lang.amberdev.online</a> (panel de administración) y pégala abajo.</p>

        <form action="options.php" method="post">
            <?php settings_fields('lang_translate_api'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="lang_translate_api_url">URL de la API</label></th>
                    <td>
                        <input type="url" id="lang_translate_api_url" name="lang_translate_api_url"
                               value="<?php echo esc_attr(get_option('lang_translate_api_url', 'https://lang.amberdev.online')); ?>"
                               class="regular-text" placeholder="https://lang.amberdev.online">
                        <p class="description">URL base del servicio (sin barra final).</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="lang_translate_api_key">API Key</label></th>
                    <td>
                        <input type="password" id="lang_translate_api_key" name="lang_translate_api_key"
                               value="<?php echo esc_attr(get_option('lang_translate_api_key', '')); ?>"
                               class="regular-text" placeholder="lt_xxxxxxxx...">
                        <p class="description">Credencial creada en el panel de Lang (header <code>X-Api-Key</code>).</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Selector de idioma flotante</th>
                    <td>
                        <label>
                            <input type="hidden" name="lang_translate_api_floating_widget" value="0">
                            <input type="checkbox" name="lang_translate_api_floating_widget" value="1"
                                <?php checked(get_option('lang_translate_api_floating_widget'), '1'); ?>>
                            Mostrar botón flotante ES | EN que traduce la página al instante
                        </label>
                        <p class="description">Aparece abajo a la derecha. El visitante elige idioma y la página se traduce automáticamente.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button(__('Guardar', 'lang-translate-api')); ?>
        </form>

        <hr>
        <h2>Uso</h2>
        <h3>Widget flotante</h3>
        <p>Si activaste <strong>Selector de idioma flotante</strong>, en el front se verá un botón ES | EN. Al hacer clic se traduce la página y se recuerda el idioma.</p>
        <h3>Shortcodes</h3>
        <ul style="list-style: disc; margin-left: 1.5em;">
            <li><code>[lang_translate content="Hello world" target="es"]</code> — traduce a español.</li>
            <li><code>[lang_translate target="en"]Texto en español aquí.[/lang_translate]</code> — traduce el contenido a inglés.</li>
            <li><code>[lang_translate_html target="es"]&lt;p&gt;Hello&lt;/p&gt;[/lang_translate_html]</code> — traduce HTML.</li>
        </ul>
        <h3>En PHP (temas o plugins)</h3>
        <pre style="background: #f0f0f1; padding: 1em; overflow-x: auto;">&lt;?php
echo lang_translate_api_translate('Hello', 'es');           // texto
echo lang_translate_api_translate_html($html, 'en');        // HTML
?&gt;</pre>
    </div>
    <?php
}
